<?php
/**
 * EDD Customer Dashboard
 *
 * @package   EDD_Customer_Dashboard
 * @author    Josh Mallard <josh@limecuda.com>
 * @license   GPL-2.0+
 * @link      http://joshmallard.com
 */

/**
 * EDD Customer Dashboard Class
 *
 * @package EDD_Customer_Dashboard
 * @author  Josh Mallard <josh@limecuda.com>
 */
class EDD_Customer_Dashboard {

	/**
	 * Plugin version, used for cache-busting of style and script file references.
	 *
	 * @since   1.0.0
	 *
	 * @var     string
	 */
	const VERSION = '1.0.0';

	/**
	 * text-domain variable
	 *
	 * @since    1.0.0
	 *
	 * @var      string
	 */
	protected $plugin_slug = 'edd_customer_dashboard';

	/**
	 * Instance of this class.
	 *
	 * @since    1.0.0
	 *
	 * @var      object
	 */
	protected static $instance = null;

	/**
	 * Initialize the plugin by setting localization and loading public scripts
	 * and styles.
	 *
	 * @since     1.0.0
	 */
	private function __construct() {

		// Load plugin text domain
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

		add_shortcode( 'edd_customer_dashboard', array( $this, 'edd_customer_dashboard_shortcode' ) );

	}

	/**
	 * Return the plugin slug.
	 *
	 * @since    1.0.0
	 *
	 * @return    Plugin slug variable.
	 */
	public function get_plugin_slug() {
		return $this->plugin_slug;
	}

	/**
	 * Return an instance of this class.
	 *
	 * @since     1.0.0
	 *
	 * @return    object    A single instance of this class.
	 */
	public static function get_instance() {

		// If the single instance hasn't been set, set it now.
		if ( null == self::$instance ) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	/**
	 * Load the plugin text domain for translation.
	 *
	 * @since    1.0.0
	 */
	public function load_plugin_textdomain() {

		$domain = $this->plugin_slug;
		$locale = apply_filters( 'plugin_locale', get_locale(), $domain );

		load_textdomain( $domain, trailingslashit( WP_LANG_DIR ) . $domain . '/' . $domain . '-' . $locale . '.mo' );
		load_plugin_textdomain( $domain, FALSE, basename( plugin_dir_path( dirname( __FILE__ ) ) ) . '/languages/' );

	}

	/**
	 * Register and enqueue public-facing style sheet.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_slug . '-plugin-styles', plugins_url( 'style.css', __FILE__ ), array(), self::VERSION );
	}

	/**
	 * Build the dashboard shortcode
	 *
	 * @since 1.0.0
	 */
	public function edd_customer_dashboard_shortcode( $atts, $content = "" ) {

		ob_start();

		if( is_user_logged_in() ) {

			
			$dashboard_content = $this->build_page_content();

			// Customer dashboard menu
			do_action( 'edd_cd_before_menu' );
			include_once ( plugin_dir_path( __FILE__ ) . '/templates/customer-dashboard-menu.php' );
			do_action( 'edd_cd_after_menu' );

			?>
			<div class="col-lg-9 col-12">
				<div class="customer-dashboard-content">
					<?php
					do_action( 'edd_cd_before_content' );

					echo $dashboard_content;

					do_action( 'edd_cd_after_content' );
					?>
				</div>
			</div>
			</div>

			<?php
		} else {
			?>
			<div class="s-product dlogin col-xs-12">
			<div class="page-con login-con col-sm-6 col-xs-12">
            <?php echo edd_login_form( get_permalink() ); ?>
             </div>
            <div class=" col-sm-6 hidden-xs">
                <img src="<?php echo get_template_directory_uri(); ?>/images/login.svg" oncontextmenu="return false" onerror="this.onerror=null; this.src='<?php echo get_template_directory_uri(); ?>/images/login.png'" alt="ورود به سایت" title="ورود به سایت" class="login-img"/>
             </div>
            </div>
		<?php }

		$dashboard = ob_get_clean();

		return $dashboard;

	}

	/**
	 * Customer dashboard page content
	 *
	 * @since	1.0.0
	 */
	public function build_page_content() {
		
        $task = !empty( $_GET[ 'task' ] ) ? $_GET[ 'task' ] : '';
		$updated = !empty( $_GET[ 'updated' ] ) ? $_GET[ 'updated' ] : '';
        
		ob_start();

		$custom = apply_filters('edd_cd_custom_task', false, $task );
		switch ( $task ) {

			case '' :
				echo '<div class="alert alert-info">از طریق گزینه های سمت راست می توانید حساب خود را مدیریت نمایید.</div>';
			break;
                
            case 'profile' :
				echo do_shortcode( '[edd_profile_editor]' );
			break;

			case 'purchases' :
				echo do_shortcode( '[purchase_history]' );
			break;

			case 'download' :
				echo do_shortcode( '[download_history]' );
			break;
            
            case 'points' :
				echo do_shortcode( '[edd_points_history]' );
			break;

			case 'license' :
				echo do_shortcode( '[edd_license_keys]' );
			break;
                
            /*case 'wishlist' :
				echo do_shortcode( '[edd_wish_lists]' );
			break;*/
                
            case 'submit-ticket' :
			echo do_shortcode( '[ticket-submit]' );
			break;
			
			case 'tickets' :
			echo do_shortcode( '[tickets]' );
			break;
                
            case 'fes_become_vendor' :
				echo do_shortcode( '[fes_registration_form]' );
			break;
			
			case 'wallet' :
				echo '<div class="wallet-text">موجودی کیف پول ' . do_shortcode( '[edd_wallet_value]'). '</div>'; 
				echo do_shortcode( '[edd_deposit]' );
			break;

			case $custom :
				do_action( 'edd_cd_custom_task_content' );
			break;

			default:
                
                echo '<div class="alert alert-info">از طریق گزینه های سمت راست می توانید حساب خود را مدیریت نمایید.</div>';

			break;

		}
        if($updated == 'true') {
            echo '<div class="alert up-alert alert-success">
				با موفقیت ذخیره شد.
			</div>';
        }
		$dashboard_content = ob_get_clean();

		return $dashboard_content;

	}

}
